/*  Terminierung der Arbeitsgänge einer ABK.

    Bei der Terminierung werden auch die Kopfkostenstellen bzw.Nebenressourcen beachtet und zusammen mit der Hauptressource
    terminiert. Auf den Nebenressourcen wird dabei eine maximale Überlastung mit einem vordefiniertem Faktor
    zugelassen.

    Die Einschänkung auf bestimmte AGe der ABK wird bei der Generierung der Vertex-Pfade an die Terminerung weitergereicht
    (Aufruf scheduling.vertex__generate_path).
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__abk_ab2__termination', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__abk_ab2__termination(
      _abk_ix integer,                            -- ID der zu terminierenden ABK.

      _timeframe_start timestamp,                 -- Minimum des Zeitbereichs, in dem die ABK einterminiert werden soll. Passt die ABK nicht in den Zeitbereich, schlägt Terminierung mit Exception fehl.
      _timeframe_end timestamp,                   -- Maximum des Zeitbereichs, in dem die ABK einterminiert werden soll. Passt die ABK nicht in den Zeitbereich, schlägt Terminierung mit Exception fehl.

      _write_to_disk bool = false,                -- Bei gesetztem Parameter "_write_to_disk" soll das Terminierungsergbenis in die Tabelle scheduling.resource_timeline geschrieben werden.
                                                  -- Unabhänging von diesem Parameter wird das Terminerungsergebnis parallele von der Funktion als Rückgabewerte ausgegeben.
      _direction varchar = 'forward',             -- Terminbierungsrichtung: Erlaubt sind "forward" (Vorwärtsterminuerung) und "backward" (Rückwärtsterminierung).

      _scenario varchar = null,                   -- TODO AXS: Unbenutzt! Kann das weg?

      _checkBlockedTimes bool = true,             -- Bei 'false' DLZ-Terminierung: Ignoriert Blockierungen auf der Ressource durch andere Task*-Einträge. Es werden nur Off-Times beachtet.

      _blocktime_refcursor refcursor = null,      -- Möglichkeit um von außen via einem refcursor zusätzliche Timeline-Einträge in die Funktion reinzugeben.

      _termination_start_time timestamp = null,   -- Startzeit des Mittelpunkt-AG  für Mittelpunktterminierung. Angefangen mit diesem Zeitpunkt wird der Mittelpunkt-AG in die angegebene Richtung terminiert.
                                                  -- Mittelpunktterminierung hier derzeit ohne Funktion, wenn auch funktionierend. Da derzeit die Verkettungen usw. zuvor bearbeitet werden und die Mittelpunktterminierung von außerhalb durch termination_execute_auto angesteuert wird. Theoretisch könnte man innerhalb einer AVOR/ABK direkt angeben, das in diesem Fall eine Mittelpunktterminierung stattfinden soll
      _termination_start_ab2_id int = null,       -- Mittelpunkt-AG für Mittelpunktterminierung. Ist der Mittelpunkt-AG gleich einer der Bereichsgrenzen, schlägt die Terminierung mit einer Exception fehl.
      _termination_range_ab2_id_start int = null, -- Minimum des Bereichs der AGe, die von der ABK terminiert werden sollen.
      _termination_range_ab2_id_end int = null,   -- Maximum des Bereichs der AGe, die von der ABK terminiert werden sollen.

      _allow_overlap bool = false,                -- Überlappungen bei der Terminierung von Arbeitsgängen innerhalb der gleichen ABK erlauben.

      _resource_id_main_fix__set bool = false,    -- Erzwinge das Fixieren der in der Terminierung gewählten Arbeitsplatzressourcen. Wird sonst nur bei _write_to_disk = true gemacht.

      _allow_fragmentation varchar = 'no',        -- Fragmeniering erlauben: 'no' - keine Fragmentierung erlauben (Standard); 'until' - Terminierung bis zur ersten Fragmentierung durchführen; 'yes' - Fragmentierung erlauben, ab Fragmentierung werden nur noch Blockierungen gebildet.

      _loglevel int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS TABLE  (
      __ab2_id integer,
      __resource_id integer,
      __slotStartDate timestamp,
      __slotEndDate timestamp,
      __type scheduling.resource_timeline_blocktype,
      __slotFactor numeric(16,6),
      __usage numeric(12,8)
  ) AS $$

  DECLARE

      _record                   record;
      _search_id                int;
      _stat                     scheduling.resource_timeline_blockstatus;
      _ab2_a2_n_range           int[];
      _ab2_a2_n_terminated      int[];
      _max_lower_end_date       timestamp;
      _min_upper_start_date     timestamp;
      _timeframe_start_adjusted timestamp;      -- Angepasster Beginn des vorgegebenen Zeitfensters
      _timeframe_end_adjusted   timestamp;      -- Angepasstes Ende des vorgegebenen Zeitfensters
      _timeframe_adjusted       bool := false;  -- Das vorgegebene Zeitfenster musste verkleinert werden, da bereits terminierte Vorgänger- oder Nachfolger-AGe hineinragen.

  BEGIN

       -- Wir betetreten die automatische Terminierung. Terminierungszeitdaten dürfen in den AG zurückgeschrieben werden.
      PERFORM execution_flag__aquire( _flagname => 'inTerminierung' );

      RAISE NOTICE '%', format(
          $call$
                    call: scheduling.resource_timeline__abk_ab2__termination(
                          _abk_ix => %L,
                          _timeframe_start => %L,
                          _timeframe_end => %L,
                          _write_to_disk => %L,
                          _direction => %L,
                          _scenario => %L,
                          _checkBlockedTimes => %L,
                          _blocktime_refcursor => %L,
                          _termination_start_time => %L,
                          _termination_start_ab2_id => %L,
                          _termination_range_ab2_id_start => %L,
                          _termination_range_ab2_id_end => %L,
                          _allow_overlap => %L,
                          _resource_id_main_fix__set => %L,
                          _allow_fragmentation => %L,
                          _loglevel => %L
                    )

                    timstamp=%L
          $call$,
          _abk_ix, _timeframe_start, _timeframe_end, _write_to_disk,
          _direction, _scenario, _checkBlockedTimes,
          _blocktime_refcursor,
          _termination_start_time, _termination_start_ab2_id,
          _termination_range_ab2_id_start, _termination_range_ab2_id_end, _allow_overlap, _resource_id_main_fix__set, _allow_fragmentation,
          _loglevel,
          clock_timestamp()::time
      );

      -- Bereich, der für die Terminierung ausgewählten Arbeitsgangnummern ermitteln.
      _ab2_a2_n_range := scheduling.abk__ab2__range_number__get( _abk_ix, _termination_range_ab2_id_start, _termination_range_ab2_id_end, _validate_range => false );

      IF _ab2_a2_n_range[1] IS null AND _ab2_a2_n_range[2] IS null THEN
        RETURN;
      END IF;

      -- Prüfung: ABK und zugehörige AGe sind terminierbar?
      /*
      IF ( 0 < scheduling.abk__is_terminatable( _abk_ix, _termination_range_ab2_id_start, _termination_range_ab2_id_end ) ) THEN
          RAISE EXCEPTION 'abk % is not terminateable in given range: [ % -> % ]!', _abk_ix, _ab2_a2_n_range[1], _ab2_a2_n_range[2];
      END IF;
      */

      -- Prüfung: Die für die Terminierung ausgewählten AGe sind noch NICHT terminiert?
      _ab2_a2_n_terminated := array_agg( a2_n )
                         FROM ab2
                        WHERE a2_ab_ix = _abk_ix
                          AND a2_n >= _ab2_a2_n_range[1]
                          AND a2_n <= _ab2_a2_n_range[2]
                              -- NOTE AXS: Diese Bedingung mit "a2_interm AND NOT a2_ende" ersetzen ist nicht möglich, da in scheduling.resource_timeline__ab2__retermination_fragmented beim Löschen der Timeline-Einträge "a2_interm" auf "true" gesetzt bleibt.
                              -- Selbst Aufruf von scheduling.abk__termination_clear() in der Reterminierungsfunktion führt zu fehlenr durch die Bedinung.
                          AND scheduling.resource_timeline__ab2__get_scheduled_time( _ab2_id => a2_id ) > 0.0;


      IF ( array_length( _ab2_a2_n_terminated, 1 ) IS NOT NULL ) THEN
          RAISE WARNING 'a2_n % of abk % are already terminated!', array_to_string( _ab2_a2_n_terminated, ',', '*' ), _abk_ix;
          RETURN;
      END IF;

      -- Es sind KEINE Überschneidungen von AGen der gleichen ABK erlaubt, ...
      IF ( NOT _allow_overlap ) THEN
          -- ... dann aktuellen AG nach Ende der Vorgänger-AG beginnen ...
          -- Beginn des Terminierungszeitfensters nach hinten verschieben, wenn vorhergehende Termineriungsblöcke ins Terminierungszeitfenster hineinragen.
          -- Dabei nur Terminierungsblöcke auf der Hauptressource beachten.
          --_max_lower_end_date :=
          --         max(a2_et)
            SELECT a2_n, a2_et
              INTO _record
              FROM ab2
             WHERE a2_ende IS false
               AND a2_interm
               AND a2_n < _ab2_a2_n_range[1]
               AND a2_ab_ix = _abk_ix
             ORDER BY
                   a2_et DESC
             LIMIT 1
          ;
          _max_lower_end_date := _record.a2_et;

          _timeframe_start_adjusted := greatest( _max_lower_end_date, _timeframe_start );

          IF ( _timeframe_start <> _timeframe_start_adjusted) THEN
            RAISE NOTICE '_timeframe_adjusted: _ab2_a2_n_range[1]:%  a2_n:%, _timeframe_start:%, _timeframe_start_adjusted:%', _ab2_a2_n_range[1], _record.a2_n, _timeframe_start, _timeframe_start_adjusted;
          END IF;

          -- ... und vor dem Start des Nachfolger-AG enden lassen.
          -- Ende des Terminierungszeitfensters nach vorn verschieben, wenn nachfolgende Termineriungsblöcke ins Terminierungszeitfenster hineinragen.
          -- Dabei nur Terminierungsblöcke auf der Hauptressource beachten.
          --_min_upper_start_date :=
          --         min( a2_at )
            SELECT a2_n, a2_at
              INTO _record
              FROM ab2
             WHERE a2_ende IS false
               AND a2_interm
               AND a2_n > _ab2_a2_n_range[2]
               AND a2_ab_ix = _abk_ix
             ORDER BY
                   a2_at ASC
             LIMIT 1
          ;
          _min_upper_start_date := _record.a2_at;

          _timeframe_end_adjusted   := least( _min_upper_start_date, _timeframe_end );

          IF ( _timeframe_end <> _timeframe_end_adjusted) THEN
            RAISE NOTICE '_timeframe_adjusted: _ab2_a2_n_range[2]:%  a2_n:%, _timeframe_end:%, _timeframe_end_adjusted:%', _ab2_a2_n_range[2], _record.a2_n, _timeframe_end, _timeframe_end_adjusted;
          END IF;

          -- Ein schon terminerter Vorgänger- oder Nachfolger-AG ragt ins vorgegebe Zeitfenster.
          IF ( _timeframe_start <> _timeframe_start_adjusted OR _timeframe_end <> _timeframe_end_adjusted ) THEN
              _timeframe_adjusted := true;
          END IF;

          -- Eine _termination_start_time für die Mittelpunktterminierung ist vorgegeben.
          IF _termination_start_time IS NOT NULL THEN
              RAISE WARNING 'mittelpunktterminierung wrong call => need to be called in resource_timeline__abk_ab2__termination__grouped'; -- TODO: Mittelpunkt komplett entfernen
              -- wir terminieren vorwärts.
              IF _direction = 'forward' THEN
                  -- Startzeitpunkt der Mittelpunktterminierung nach hinten verschieben, wenn vorhergehende Termineriungsblöcke ins Terminierungszeitfenster hineinragen.
                  -- Dabei nur Terminierungsblöcke auf der Hauptressource beachten.
                  _termination_start_time := greatest( _max_lower_end_date, _termination_start_time );
              -- Wir terminieren rückwärts.
              ELSE
                  -- Startzeitpunkt der Mittelpunktterminierung nach vorne verschieben, wenn nachfolgende Termineriungsblöcke vor der Terminerungsstartzeit beginnen.
                  -- Dabei nur Terminierungsblöcke auf der Hauptressource beachten.
                  IF _min_upper_start_date < _termination_start_time THEN
                      _termination_start_time := _timeframe_start_adjusted;
                  END IF;
              END IF;
          END IF;

      -- Es sind Überschneidungen von AGen der gleichen ABK vollständig erlaubt, ...
      ELSE
      /*  -- Überschneidungen nur zum Teil erlaubt, nur Überschneidungen mit direktem Vorgänger bzw. direkter Nachfolger-AG sind erlaubt.
          -- ... dann Start des aktuellen AG zum Start des Vorgänger-AG beginnen ...
          -- Beginn des Terminierungszeitfensters nach hinten verschieben, wenn vorhergehende Termineriungsblöcke ins Terminierungszeitfenster hineinragen.
          _timeframe_start_adjusted :=
              greatest( min( rt.ti_date_start ), _timeframe_start )
              FROM ab2
              JOIN scheduling.resource_timeline rt on rt.ti_a2_id = ab2.a2_id
              WHERE
                ab2.a2_ende IS false --scheduling.ab2__is_terminateable( ab2, _loglevel => _loglevel )
                AND ab2.a2_n < _ab2_a2_n_range[1]
                AND ab2.a2_ab_ix = _abk_ix
          ;
          -- ... und zum Ende des Nachfolger-AG beenden.
          -- Ende des Terminierungszeitfensters nach vorn verschieben, wenn nachfolgende Termineriungsblöcke ins Terminierungszeitfenster hineinragen.
          _timeframe_end_adjusted :=
              least( max( rt.ti_date_end ), _timeframe_end )
              FROM ab2
              JOIN scheduling.resource_timeline rt on rt.ti_a2_id = ab2.a2_id
              WHERE
                ab2.a2_ende IS false --scheduling.ab2__is_terminateable( ab2, _loglevel => _loglevel )
                AND ab2.a2_n > _ab2_a2_n_range[2]
                AND ab2.a2_ab_ix = _abk_ix
          ;
      */
          -- ... dann die über die Parameter "_timeframe_start" und "_timeframe_end" Zeitgrenzen verwenden.
          _timeframe_start_adjusted := _timeframe_start;
          _timeframe_end_adjusted   := _timeframe_end;
      END IF;

      IF (
           _termination_start_time < _timeframe_start_adjusted
        OR _termination_start_time > _timeframe_end_adjusted
       ) THEN
          RAISE EXCEPTION 'Termination startime is out of bounds: % [ % -> % ]', _termination_start_time, _timeframe_start_adjusted, _timeframe_end_adjusted;
      END IF;

      -- build vertices
      PERFORM scheduling.vertex__generate_path( _abk_ix, _termination_range_ab2_id_start, _termination_range_ab2_id_end, _loglevel => _loglevel );

      -- cleanup old results
      DELETE FROM scheduling.resource_termination_results WHERE abk_ab_ix = _abk_ix;

      -- termin search. vertex contains only ab2 in range
      _search_id := scheduling.resource_timeline__termination_search(
                        _abk_ix                    => _abk_ix,
                        _timeframe_start           => _timeframe_start_adjusted,
                        _timeframe_end             => _timeframe_end_adjusted,
                        _direction                 => _direction,
                        _scenario                  => _scenario,
                        _checkBlockedTimes         => _checkBlockedTimes,
                        _blocktime_refcursor       => _blocktime_refcursor,
                        _termination_start_time    => _termination_start_time,
                        _termination_start_ab2_id  => _termination_start_ab2_id,
                        _allow_overlap             => _allow_overlap,
                        _timeframe_adjusted        => _timeframe_adjusted,
                        _allow_fragmentation       => _allow_fragmentation,
                        _loglevel                  => _loglevel
                    )
        ;

      -- Status bei DLZ setzen.
      IF _checkBlockedTimes = false THEN
          _stat := 'dlz'::scheduling.resource_timeline_blockstatus;
      END IF;

      -- Zwischenspeichern der Terminierungsergbenisse in einer temporärem Tabelle, damit die dafür notwendige Abfrage auch bei
      -- gleichzeitiger Speicherung und Rückgabe nur einmal aufgeführt werden muss.
      -- Temporäre Tabelle erstellen, falls diese noch fehlt.

      SET client_min_messages = warning; -- hide notice already exists

      CREATE TEMP TABLE IF NOT EXISTS _tmp_termination_result (
          ttr_id              serial PRIMARY KEY,
          ttr_search_id       integer,
          ttr_ab2_id          integer,
          ttr_resource_id     integer,
          ttr_slotStartDate   timestamp,
          ttr_slotEndDate     timestamp,
          ttr_type            scheduling.resource_timeline_blocktype,
          ttr_slotFactor      numeric(16,6),
          ttr_usage           numeric(12,8)
      );

      SET client_min_messages = notice;

      -- Eventuell enthaltene vorherige Einträge mit gleicher SuchID löschen. (Sollte es nie geben.)
      DELETE FROM _tmp_termination_result
       WHERE ttr_search_id = _search_id;
      -- Terminierungsergbenisse zwischenspeichern.
      INSERT INTO _tmp_termination_result ( ttr_search_id, ttr_ab2_id, ttr_resource_id , ttr_slotStartDate , ttr_slotEndDate , ttr_type     , ttr_slotFactor , ttr_usage  )
           -- Hauptressource
           SELECT                                _search_id   , _ab2_id   , _resource_id    , _slotStartDate    , _slotEndDate    , _slotType    , _slotFactor    , _usage
             FROM scheduling.resource_termination_results__abk__fastest_fetch( _abk_ix, _search_id )
           --
            UNION
           -- Kopfkostenstelle
           SELECT                                _search_id   , _ab2_id   , _top_resource_id, _top_slotStartDate, _top_slotEndDate, _top_slotType, _top_slotFactor, _top_usage
             FROM scheduling.resource_termination_results__abk__fastest_fetch( _abk_ix, _search_id )
            WHERE _top_resource_id IS NOT null;

      -- Terminierungsergbenisse in Tabelle resource_timeline speichern.
      IF _write_to_disk IS true THEN
          INSERT INTO scheduling.resource_timeline(        ti_a2_id  , ti_resource_id  , ti_date_start     , ti_date_end     , ti_type , ti_ta_kf       , ti_usage , ti_stat  )
               SELECT                                           ttr_ab2_id, ttr_resource_id , ttr_slotStartDate , ttr_slotEndDate , ttr_type, ttr_slotFactor , ttr_usage, _stat
                 FROM _tmp_termination_result
                WHERE ttr_search_id = _search_id;

          -- Die zur Terminierung verwendete Hauptressource pro AG in der Spalte "a2w_resource_id_main_terminated" merken.
          -- Zusätzlich die verwendete Hauptressource als Vorgabe für die nächste Terminierungen des entsprechenden AGs in
          -- Spalte "a2w_resource_id_main_fix" hinterlegen. Soll eine andere Ressource bei der nächsten Termineriung verwendet
          -- werden, muss aus der Oberfläche heraus die Vorgabe entsprechend angepasst werden.
          WITH
              _result AS (
                  -- hole aktuelles Terminierungsergebnis - ALLE terminierten ab2
                  SELECT DISTINCT ttr_ab2_id AS a2_id
                       , scheduling.resource_timeline__resource_id_main_resource__get( ttr_ab2_id ) AS main_resource_id
                    FROM _tmp_termination_result
                   WHERE ttr_search_id = _search_id
              )
              UPDATE ab2_wkstplan
                SET a2w_resource_id_main_terminated = _result.main_resource_id
                  , a2w_resource_id_main_fix = _result.main_resource_id
              FROM _result -- ACHTUNG: mehrere Datensätze, ALLE terminierten ab2
              WHERE a2w_a2_id = _result.a2_id
                AND a2w_marked >= 0;

          -- Bei ohne Konflikt DLZ-terminierten AGen die DLZ-Markierung wieder löschen.
          IF _checkBlockedTimes IS false THEN
              WITH
                _terminated_ab2 AS (
                    SELECT DISTINCT a2_id
                      FROM _tmp_termination_result
                      JOIN ab2 ON a2_id = ttr_ab2_id
                      JOIN ab2_wkstplan ON a2w_a2_id = a2_id
                     WHERE ttr_search_id = _search_id
                       AND ttr_resource_id = a2w_resource_id_main_terminated
                       AND ttr_type IN ( 'task', 'task.buffer' )
                ),
                _not_conflicted_ab2 AS (
                    SELECT a.a2_id, b
                      FROM _terminated_ab2 a
                      JOIN LATERAL scheduling.resource_timeline__overlap_duration__by__a2_id(
                                _a2_id                => a.a2_id
                              , _filter_for_other_ab2 => 'none'   -- DLZ-Markierung nur löschen, wenn gar keine Überlappungen mit anderen AGen existieren.
                            ) b ON TRUE
                      WHERE b = 0
                )
                UPDATE scheduling.resource_timeline ti
                    SET ti_stat = null
                  FROM _not_conflicted_ab2 nc
                  WHERE ti_a2_id = nc.a2_id
                    AND ti_stat = 'dlz';
          END IF;
      -- _write_to_disk = false
      ELSE
          -- Das Fixieren, der in der Terminierung gewählten Arbeitsplatzressourcen ist erzwungen...
          IF _resource_id_main_fix__set IS true THEN
              -- ... dann fixiere die in der Terminierung gewählten Arbeitsplatzressourcen.
              WITH _result AS (
                  -- hole ALLE terminierten ab2
                  SELECT DISTINCT ttr.ttr_ab2_id AS a2_id, ttr.ttr_resource_id AS resource_id
                    FROM _tmp_termination_result AS ttr
                    JOIN scheduling.resource ra ON ra.context = 'ab2'                             -- AG-Ressource
                                               AND ra.context_id = ttr.ttr_ab2_id
                    JOIN scheduling.resource_requirement rr ON rr.context = 'ksvba'
                                                           AND rr.required_by = ra.id
                    JOIN scheduling.resource_requirement_option rro ON rro.requirement_id = rr.id
                    JOIN scheduling.resource rk ON rk.context = 'ksvba'                           -- KSVBA-Ressource (Hauptressource)
                                               AND rk.id = rro.resource_id
                                               AND rk.id = ttr.ttr_resource_id
                   WHERE ttr.ttr_search_id = _search_id
              )
              UPDATE ab2_wkstplan
                SET a2w_resource_id_main_fix = _result.resource_id
                FROM _result -- ACHTUNG: mehrere Datensätze, ALLE terminierten ab2
              WHERE a2w_a2_id = _result.a2_id
                AND a2w_marked >= 0;
          END IF;
      END IF;

      -- Wir verlassen die automatische Terminierung wieder.
      PERFORM execution_flag__release( _flagname => 'inTerminierung' );

      -- Terminierungsergbenisse zurückgeben.
      RETURN QUERY
          -- Hauptressource
          SELECT                                           ttr_ab2_id, ttr_resource_id , ttr_slotStartDate , ttr_slotEndDate , ttr_type, ttr_slotFactor , ttr_usage
            FROM _tmp_termination_result
           WHERE ttr_search_id = _search_id;

    END $$ LANGUAGE plpgsql;
    --